package gov.cms.grouper.snf.util;

import gov.cms.grouper.snf.component.v100.TestUtil;
import gov.cms.grouper.snf.lego.SnfUtils;
import gov.cms.grouper.snf.lego.TriFunction;
import gov.cms.grouper.snf.lego.Triple;
import gov.cms.grouper.snf.model.Assessment;
import gov.cms.grouper.snf.model.SnfDiagnosisCode;
import gov.cms.grouper.snf.model.reader.Rai300;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Set;
import java.util.function.BiFunction;
import org.junit.jupiter.api.Assertions;
import org.junit.jupiter.api.Test;

public class ClaimInfoTest {

  public static final int version = 100;

  public static final List<Assessment> list =
      Arrays.asList(new Assessment(Rai300.B0100.getXmlTag(), Rai300.B0100.name(), 3),
          new Assessment(Rai300.D0300.getXmlTag(), Rai300.D0300.name(), 1),
          new Assessment(Rai300.GG0170E1.getXmlTag(), Rai300.GG0170E1.name(), Assessment.CHECKED));


  public static final BiFunction<String, Triple<String, String, String>, SnfDiagnosisCode> codeBuilder =
      (value, cats) -> {
        SnfDiagnosisCode code =
            new SnfDiagnosisCode(value, cats.getOne(), cats.getTwo(), cats.getThree());
        return code;
      };

  public static final List<SnfDiagnosisCode> codes =
      Arrays.asList(ClaimInfoTest.codeBuilder.apply("v1", Triple.of("cc1", "sp1", "nc1")),
          ClaimInfoTest.codeBuilder.apply("v2", Triple.of("cc2", "sp2", "nc2")),
          ClaimInfoTest.codeBuilder.apply("v3", Triple.of("cc3", "sp3", "nc3")),
          ClaimInfoTest.codeBuilder.apply("v4", Triple.of("cc4", "sp4", "nc4")));

  @Test
  public void testGetAssessmentValue() {
    Rai300 item = Rai300.B0100;
    int expected = 3;
    ClaimInfo claim = ClaimInfo.of(version, false, ClaimInfoTest.list);
    int actual = claim.getAssessmentValue(item);
    Assertions.assertEquals(expected, actual);

    item = Rai300.GG0170E1;
    expected = 1;
    actual = claim.getAssessmentValue(item);
    Assertions.assertEquals(expected, actual);

    item = Rai300.I2500;
    expected = Assessment.NULL_VALUE;
    actual = claim.getAssessmentValue(item);
    Assertions.assertEquals(expected, actual);

  }


  @Test
  public void testGetAssessment() {
    Rai300 item = Rai300.B0100;
    String expected = item.name();
    ClaimInfo claim = ClaimInfo.of(version, false, ClaimInfoTest.list);
    String actual = claim.getAssessment(item).getItem();
    Assertions.assertEquals(expected, actual);

    item = Rai300.GG0170E1;
    expected = item.name();
    actual = claim.getAssessment(item).getItem();
    Assertions.assertEquals(expected, actual);

    item = Rai300.I2500;
    Assessment result = claim.getAssessment(item);
    Assertions.assertNull(result);

  }

  @Test
  public void testIsAnyAssessmentValuesPresent() {
    Set<Rai300> rai300s = SnfUtils.toSet(Rai300.GG0130C5, Rai300.GG0170E1);
    ClaimInfo claim = ClaimInfo.of(version, false, ClaimInfoTest.list);
    boolean expected = true;
    boolean actual = claim.isAnyAssessmentValuesPresent(rai300s);
    Assertions.assertEquals(expected, actual);

    rai300s = SnfUtils.toSet(Rai300.GG0130C5, Rai300.B0100);
    expected = false;
    actual = claim.isAnyAssessmentValuesPresent(rai300s);
    Assertions.assertEquals(expected, actual);

    rai300s = SnfUtils.toSet(Rai300.K0100C);
    expected = false;
    actual = claim.isAnyAssessmentValuesPresent(rai300s);
    Assertions.assertEquals(expected, actual);

    expected = false;
    actual = claim.isCheckedAndNotNull(Rai300.O0500G);
    Assertions.assertEquals(expected, actual);

    expected = true;
    actual = claim.isCheckedAndNotNull(Rai300.GG0170E1);
    Assertions.assertEquals(expected, actual);
  }

  @Test
  public void testCountAssessmentPresent() {
    ClaimInfo claim = ClaimInfo.of(version, false, ClaimInfoTest.list);
    Set<Rai300> rai300s = SnfUtils.toSet(Rai300.GG0130C5, Rai300.GG0170E1);
    int expected = 1;
    int actual = claim.countAssessmentPresent(rai300s);
    Assertions.assertEquals(expected, actual);

    rai300s = SnfUtils.toSet(Rai300.D0300, Rai300.GG0170E1, Rai300.O0500E);
    expected = 2;
    actual = claim.countAssessmentPresent(rai300s);
    Assertions.assertEquals(expected, actual);

    rai300s = SnfUtils.toSet(Rai300.B0100, Rai300.O0500E);
    expected = 0;
    actual = claim.countAssessmentPresent(rai300s);
    Assertions.assertEquals(expected, actual);

  }

  @Test
  public void testGetAssessmentNames() {
    ClaimInfo claim = ClaimInfo.of(version, false, ClaimInfoTest.list);
    Set<String> expected = SnfUtils.toSet(Rai300.D0300.getXmlTag(), Rai300.GG0170E1.getXmlTag(),
        Rai300.B0100.getXmlTag());
    Set<String> actual = claim.getAssessmentNames();
    Assertions.assertEquals(expected, actual);
  }

  /*
   * @Test public void testGetValues() { BiFunction<String, String, SnfDiagnosisCode> mkCode =
   * (value, nc) -> { SnfDiagnosisCode code = new SnfDiagnosisCode(value, null, null, nc); return
   * code; };
   *
   * List<SnfDiagnosisCode> codes = Arrays.asList(mkCode.apply("1", "Cystic Fibrosis"),
   * mkCode.apply("2", "Lung Transplant Status")); Set<String> expected = SnfUtils.toSet("1", "2");
   * Set<String> actual = ClaimInfo.getValues(codes); Assertions.assertEquals(expected, actual);
   *
   * expected = SnfUtils.toSet("v1", "v2", "v3", "v4"); actual =
   * ClaimInfo.getValues(ClaimInfoTest.codes); Assertions.assertEquals(expected, actual); }
   */

  @Test
  public void testGetNtaCategories() {
    BiFunction<String, String, SnfDiagnosisCode> mkCode = (value, nc) -> {
      SnfDiagnosisCode code = new SnfDiagnosisCode(value, null, null, nc);
      return code;
    };

    List<SnfDiagnosisCode> codes = Arrays.asList(mkCode.apply("1", "Cystic Fibrosis"),
        mkCode.apply("2", "Lung Transplant Status"));
    Set<String> expected = SnfUtils.toSet("Cystic Fibrosis", "Lung Transplant Status");
    Set<String> actual =
        new ClaimInfo(version, false, Collections.emptyList()).getNtaCategories(codes);
    Assertions.assertEquals(expected, actual);

  }

  @Test
  public void testPerformanceRecode() {
    Assertions.assertEquals(4,
        new ClaimInfo(version, false, Collections.emptyList()).performanceRecode(() -> 6));
    Assertions.assertEquals(1,
        new ClaimInfo(version, false, Collections.emptyList()).performanceRecode(() -> 2));
    Assertions.assertEquals(0,
        new ClaimInfo(version, false, Collections.emptyList()).performanceRecode(() -> 1));
  }

  @Test
  public void testIsComaAndNoActivities() {
    Assessment gg0130a1 = new Assessment("GG0130A1", "GG0130A1", 1);
    Assessment gg0130c1 = new Assessment("GG0130C1", "GG0130C1", 9);
    Assessment gg0170b1 = new Assessment("GG0170B1", "GG0170B1", 88);
    Assessment gg0170c1 = new Assessment("GG0170C1", "GG0170C1", 88);
    Assessment gg0170d1 = new Assessment("GG0170D1", "GG0170D1", 9);
    Assessment gg0170e1 = new Assessment("GG0170E1", "GG0170E1", 1);
    Assessment gg0170f1 = new Assessment("GG0170F1", "GG0170F1", 88);
    List<Assessment> list =
        Arrays.asList(gg0130a1, gg0130c1, gg0170b1, gg0170c1, gg0170d1, gg0170e1, gg0170f1);
    ClaimInfo claim = ClaimInfo.of(version, false, list);
    Assertions.assertTrue(claim.isComaAndNoActivities(() -> 1));
    Assertions.assertFalse(claim.isComaAndNoActivities(() -> 0));

    Assessment gg0130a5 = new Assessment("GG0130A5", "GG0130A5", 1);
    Assessment gg0130c5 = new Assessment("GG0130C5", "GG0130C5", 9);
    Assessment gg0170b5 = new Assessment("GG0170B5", "GG0170B5", 88);
    Assessment gg0170c5 = new Assessment("GG0170C5", "GG0170C5", 88);
    Assessment gg0170d5 = new Assessment("GG0170D5", "GG0170D5", 9);
    Assessment gg0170e5 = new Assessment("GG0170E5", "GG0170E5", 1);
    Assessment gg0170f5 = new Assessment("GG0170F5", "GG0170F5", 88);
    list = Arrays.asList(gg0130a5, gg0130c5, gg0170b5, gg0170c5, gg0170d5, gg0170e5, gg0170f5);
    claim = ClaimInfo.of(version, true, list);
    Assertions.assertTrue(claim.isComaAndNoActivities(() -> 1));
    Assertions.assertFalse(claim.isComaAndNoActivities(() -> 0));
  }

  @Test
  public void testCheckedAndNotNull() {
    ClaimInfo claim = ClaimInfo.of(version, false, ClaimInfoTest.list);
    boolean expected = true;
    boolean actual = claim.isCheckedAndNotNull(Rai300.GG0170E1);
    Assertions.assertEquals(expected, actual);

    expected = false;
    actual = claim.isCheckedAndNotNull(Rai300.B0100);
    Assertions.assertEquals(expected, actual);

    expected = false;
    actual = claim.isCheckedAndNotNull(Rai300.GG0170B1);
    Assertions.assertEquals(expected, actual);

  }

  @Test
  public void testHasAssessmentOf() {
    ClaimInfo claim = ClaimInfo.of(version, false, ClaimInfoTest.list);
    boolean expected = true;
    boolean actual = claim.hasAssessmentOf(Rai300.B0100, (item) -> item.getValueInt() == 3);
    Assertions.assertEquals(expected, actual);

    expected = false;
    actual = claim.hasAssessmentOf(Rai300.I5500, (item) -> item.getValueInt() == 3);
    Assertions.assertEquals(expected, actual);

  }


  @Test
  public void testIsClassifiedBehavioralSymptomsCognitivePerformance() {

  }


  @Test
  public void testCalculateFunctionScore() {
    Set<Rai300> rais = SnfUtils.concat(SnfUtils.toSet(Rai300.A0310B, Rai300.E0100A, Rai300.B0100),
        SnfUtils.toSet(Rai300.B0700, Rai300.C0100, Rai300.C0500),
        SnfUtils.toSet(Rai300.C0700, Rai300.C1000, Rai300.D0300),
        SnfUtils.toSet(Rai300.D0600, Rai300.E0100A, Rai300.E0100B));
    List<Assessment> assessments = TestUtil.getAll(rais, 6);

    ClaimInfo info = ClaimInfo.of(version, false, assessments);

    int expected = 21;
    int actual = info.calculateFunctionScore((ver, assessment, s) -> 3,
        SnfUtils.toSet(Rai300.A0310B, Rai300.E0100A, Rai300.B0100),
        SnfUtils.toSet(Rai300.B0700, Rai300.C0100, Rai300.C0500),
        SnfUtils.toSet(Rai300.C0700, Rai300.C1000, Rai300.D0300),
        SnfUtils.toSet(Rai300.D0600, Rai300.E0100A, Rai300.E0100B));
    Assertions.assertEquals(expected, actual);

    TriFunction<Integer, ClaimInfo, String, Integer> lastNonZeroInteger =
        (ver, assessment, item) -> {
          Integer num = null;
          for (int i = item.length() - 1; i >= 0; i--) {
            char c = item.charAt(i);
            int v = c;
            if (v >= 49 && v <= 57) {
              num = SnfUtils.parse(c + "", null);
              break;
            }
          }
          return num;
        };

    expected = 19;

    actual = info.calculateFunctionScore(lastNonZeroInteger,
        SnfUtils.toSet(Rai300.A0310B, Rai300.E0100A, Rai300.B0100),
        SnfUtils.toSet(Rai300.B0700, Rai300.C0100, Rai300.C0500),
        SnfUtils.toSet(Rai300.C0700, Rai300.C1000, Rai300.D0300),
        SnfUtils.toSet(Rai300.D0600, Rai300.E0100A, Rai300.E0100B));
    Assertions.assertEquals(expected, actual);

  }


}
